from flask import Flask, render_template, request, jsonify, redirect, url_for, flash
from flask_socketio import SocketIO, emit, join_room
from flask_login import LoginManager, UserMixin, login_user, login_required, logout_user, current_user
import joblib
import numpy as np
import sqlite3
import uuid 
from werkzeug.security import generate_password_hash, check_password_hash
from datetime import datetime
import time

app = Flask(__name__)
app.config['SECRET_KEY'] = 'master-key-saas-2025'
socketio = SocketIO(app, cors_allowed_origins="*")

# --- AUTH SETUP ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login'

# Load AI
try:
    model = joblib.load('guardian_brain.pkl')
except:
    print("⚠️ AI Model missing. Run train_brain.py to fix.")

# --- MEMORY SYSTEMS ---
failed_attempts = {}    # Tracks failures per CARD
blocked_cards = set()   # List of banned CARDS
client_card_usage = {}  # Tracks CARDS per USER SESSION (Card Cycling)

# --- DATABASE SETUP ---
def get_db_connection():
    conn = sqlite3.connect('system.db')
    conn.row_factory = sqlite3.Row
    return conn

def init_db():
    conn = get_db_connection()
    c = conn.cursor()
    # Users Table
    c.execute('''CREATE TABLE IF NOT EXISTS users 
                 (id INTEGER PRIMARY KEY, username TEXT UNIQUE, password TEXT, 
                  api_key TEXT UNIQUE, company_name TEXT, is_active INTEGER DEFAULT 1)''')
    
    # Logs Table
    c.execute('''CREATE TABLE IF NOT EXISTS logs 
                 (id INTEGER PRIMARY KEY, api_key TEXT, company_name TEXT, amount REAL, 
                  prediction TEXT, timestamp TEXT, reason TEXT)''')
    conn.commit()
    conn.close()

init_db()

# --- USER CLASS (FIXED) ---
class User(UserMixin):
    def __init__(self, id, username, api_key, company_name, is_active):
        self.id = id
        self.username = username
        self.api_key = api_key
        self.company_name = company_name
        self.active_status = is_active

    @property
    def is_active(self):
        return self.active_status == 1

@login_manager.user_loader
def load_user(user_id):
    conn = get_db_connection()
    user = conn.execute("SELECT * FROM users WHERE id = ?", (user_id,)).fetchone()
    conn.close()
    if user:
        return User(user['id'], user['username'], user['api_key'], user['company_name'], user['is_active'])
    return None

# --- ROUTES ---

@app.route('/')
def home():
    return render_template('home.html')

@app.route('/docs')
def documentation():
    return render_template('documentation.html')

@app.route('/super/login')
def super_login():
    return render_template('super_admin.html')

@app.route('/api/all_users')
def get_all_users():
    conn = get_db_connection()
    users = conn.execute("SELECT id, company_name, api_key, is_active FROM users").fetchall()
    conn.close()
    return jsonify([dict(row) for row in users])

@app.route('/api/toggle_ban/<int:user_id>', methods=['POST'])
def toggle_ban(user_id):
    conn = get_db_connection()
    curr = conn.cursor()
    user = curr.execute("SELECT is_active FROM users WHERE id = ?", (user_id,)).fetchone()
    new_status = 0 if user['is_active'] == 1 else 1
    curr.execute("UPDATE users SET is_active = ? WHERE id = ?", (new_status, user_id))
    conn.commit()
    conn.close()
    return jsonify({'success': True, 'new_status': new_status})

@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        company = request.form['company']
        new_api_key = "sk_live_" + str(uuid.uuid4())[:8]
        hashed_pw = generate_password_hash(password, method='scrypt')

        try:
            conn = get_db_connection()
            conn.execute("INSERT INTO users (username, password, api_key, company_name, is_active) VALUES (?, ?, ?, ?, 1)",
                         (username, hashed_pw, new_api_key, company))
            conn.commit()
            conn.close()
            return redirect(url_for('login'))
        except:
            return "Username already exists!"
    return render_template('register.html')

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        conn = get_db_connection()
        user = conn.execute("SELECT * FROM users WHERE username = ?", (username,)).fetchone()
        conn.close()
        
        if user and check_password_hash(user['password'], password):
            if user['is_active'] == 0:
                return "<h1>🚫 ACCOUNT SUSPENDED by System Administrator.</h1>"
            
            user_obj = User(user['id'], user['username'], user['api_key'], user['company_name'], user['is_active'])
            login_user(user_obj)
            return redirect(url_for('portal'))
        return "Invalid credentials"
    return render_template('login.html')

@app.route('/portal')
@login_required
def portal():
    if not current_user.is_active: 
        logout_user()
        return "<h1>🚫 Account Suspended</h1>"
    return render_template('portal.html', api_key=current_user.api_key, company=current_user.company_name)

@app.route('/logout')
@login_required
def logout():
    logout_user()
    return redirect(url_for('login'))

# --- SOCKETS & INTELLIGENCE LOGIC ---

@socketio.on('join_super_room')
def join_super():
    join_room('super_admin')

@socketio.on('join_personal_room')
def handle_join(data):
    join_room(data['api_key'])

@socketio.on('scan_transaction')
def handle_scan(data):
    # 1. AUTHENTICATE
    client_key = data.get('api_key')
    conn = get_db_connection()
    user = conn.execute("SELECT * FROM users WHERE api_key = ?", (client_key,)).fetchone()
    conn.close()

    if not user:
        emit('final_result', {'status': 'ERROR', 'msg': 'Invalid API Key'}, room=request.sid)
        return

    # KILL SWITCH
    if user['is_active'] == 0:
        emit('status_update', {'msg': '❌ API KEY SUSPENDED'}, room=request.sid)
        time.sleep(1)
        emit('final_result', {'status': 'ERROR', 'msg': 'SERVICE BANNED'}, room=request.sid)
        return

    company_name = user['company_name']
    emit('status_update', {'msg': f'Authenticating {company_name}...'}, room=request.sid)
    time.sleep(0.5)

    # --- INPUTS ---
    pay_amount = float(data['amount'])      
    real_price = float(data.get('real_price', pay_amount)) 
    card_num = data.get('card_number')
    t_type = int(data['type'])
    hour = int(data['hour'])
    m_risk = int(data['merch_risk'])
    user_sid = request.sid

    # Behavioral Data (Defaults to 0)
    b_edits = data.get('behavior_edits', 0)
    b_pastes = data.get('behavior_pastes', 0)

    result = "SAFE"
    reason = "Verified Transaction"

    # --- 1. CARD CYCLING (Session Tracking) ---
    if user_sid not in client_card_usage:
        client_card_usage[user_sid] = set()
    client_card_usage[user_sid].add(card_num)
    
    # If they use more than 2 unique cards, BLOCK.
    if len(client_card_usage[user_sid]) > 2:
        result = "FRAUD"
        reason = "Card Cycling: Tried 3+ different cards"

    # --- 2. BEHAVIORAL BIOMETRICS ---
    elif b_edits > 2:
        result = "FRAUD"
        reason = f"Behavioral Alert: Nervous Typing ({b_edits} edits)"
    
    elif b_pastes > 0:
        result = "FRAUD"
        reason = "Behavioral Alert: Card Number Pasted (Bot Behavior)"

    # --- 3. BLACKLIST & PRICE (ZERO TOLERANCE) ---
    elif card_num in blocked_cards:
        result = "FRAUD"; reason = "Global Blacklist Hit"
        
    elif pay_amount != real_price:
        result = "FRAUD"
        if pay_amount < real_price:
            reason = f"Price Manipulation: Underpayment (${pay_amount} vs ${real_price})"
        else:
            reason = f"Suspicious Overpayment (${pay_amount} vs ${real_price})"
    
    # --- 4. AI CHECK ---
    else:
        features = np.array([[pay_amount, hour, t_type, m_risk]])
        try: pred = model.predict(features)[0]
        except: pred = 0
        if t_type == 1: pred = 1; reason = "High Risk Payment Method"
        if pred == 1: result = "FRAUD"; reason = "AI Risk Detection"

    # --- SAVE ---
    conn = get_db_connection()
    conn.execute("INSERT INTO logs (api_key, company_name, amount, prediction, timestamp, reason) VALUES (?, ?, ?, ?, ?, ?)",
                 (client_key, company_name, pay_amount, result, datetime.now().strftime("%H:%M:%S"), reason))
    conn.commit()
    conn.close()

    # Notify Dashboards
    socketio.emit('new_log', {'time': datetime.now().strftime("%H:%M:%S"), 'amount': pay_amount, 'status': result, 'reason': reason}, room=client_key)
    socketio.emit('new_log', {'time': datetime.now().strftime("%H:%M:%S"), 'company': company_name, 'amount': pay_amount, 'status': result, 'reason': reason}, room='super_admin')

    if result == "FRAUD":
        fails = failed_attempts.get(card_num, 0) + 1
        failed_attempts[card_num] = fails
        if fails > 2: blocked_cards.add(card_num)
        emit('final_result', {'status': 'FRAUD', 'msg': 'Declined by Security'}, room=request.sid)
    else:
        emit('final_result', {'status': 'SAFE', 'msg': 'Approved'}, room=request.sid)

if __name__ == '__main__':
    print("⚡ FRAUD GUARDIAN PLATFORM STARTED")
    socketio.run(app, debug=True, port=5000)